import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { generatePasswordResetToken, getPasswordResetExpiration } from "@/lib/passwordReset";
import { sendPasswordResetEmail } from "@/lib/email";
import { z } from "zod";

const ForgotPasswordSchema = z.object({
  email: z.string().email(),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  if (req.method !== "POST") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  try {
    const parsed = ForgotPasswordSchema.safeParse(req.body);
    if (!parsed.success) {
      return res.status(400).json({
        ok: false,
        error: "Invalid email address",
        details: parsed.error.flatten(),
      });
    }

    const { email } = parsed.data;

    // Find user by email
    const user = await prisma.user.findUnique({
      where: { email: email.toLowerCase() },
      select: { id: true, email: true, name: true },
    });

    // Check if account exists
    if (!user) {
      return res.status(200).json({
        ok: true,
        accountExists: false,
        message: "No account found with this email address. Please create an account first.",
      });
    }

    // Generate reset token (1 hour expiration for password reset requests)
    const resetToken = generatePasswordResetToken();
    const resetExpires = getPasswordResetExpiration(1);

    // Save token to database
    await prisma.user.update({
      where: { id: user.id },
      data: {
        passwordResetToken: resetToken,
        passwordResetExpires: resetExpires,
      },
    });

    // Send password reset email (don't block on email failure)
    try {
      await sendPasswordResetEmail(user.email, user.name, resetToken);
    } catch (emailError) {
      console.error("Failed to send password reset email:", emailError);
      return res.status(500).json({
        ok: false,
        error: "Failed to send password reset email. Please try again later.",
      });
    }

    return res.status(200).json({
      ok: true,
      accountExists: true,
      message: "Password reset link has been sent to your email.",
    });
  } catch (error: any) {
    console.error("Error processing password reset request:", error);
    return res.status(500).json({
      ok: false,
      error: "Failed to process password reset request",
      details: error?.message,
    });
  }
}

